%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%                                                  %%
%%           This is file 'geoacta.cls'             %%
%%                                                  %%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%
%% Credits: Juan I. Sabbione and Danilo R. Velis, 2024.
%%
%% IMPORTANT: To compile, use pdflatex. Latex + dvipdf may work, but could
%% cause issues with figures.
%%
%% This class has been tested on Debian using:
%% pdflatex Version 3.14159265-2.6-1.40.21 (TeX Live 2020/Debian)
%%
%% Based on the standard LaTeX "book" class with the following additional packages:
%%
%%    - inputenc        : UTF-8 encoding support for proper character handling.
%%    - geometry        : Configures page layout, dimensions, and header/footer spacing.
%%    - graphicx        : Provides commands for including images.
%%    - amsmath         : Supports advanced math typesetting.
%%    - amsfonts        : Offers additional math fonts.
%%    - amssymb         : Supplies more mathematical symbols.
%%    - url             : Allows the inclusion of URLs.
%%    - hyperref        : Enables hyperlinks in the document.
%%    - babel           : Supports multilingual documents; Spanish is set as default.
%%    - xcolor          : Adds extra color options, especially for hyperlink colors.
%%    - enumerate       : Extends enumeration options.
%%    - sidecap         : Positions captions on the sides of figures or tables.
%%    - tgheros         : Sets a sans-serif font similar to Helvetica.
%%    - sansmath        : Ensures sans-serif font usage in math mode.
%%    - titlesec        : Customizes section and subsection title formatting.
%%    - natbib          : Manages citations and references with geoacta.bst style.
%%    - caption         : Controls appearance of figure and table captions.
%%    - fancyhdr        : Configures headers and footers.
%%    - lastpage        : References the last page of the document.
%%    - lineno          : Adds line numbers, useful for drafts and reviews.
%%    - etoolbox        : Offers tools for modifying commands.
%%    - indentfirst     : Ensures first paragraph indentation after section headings.
%%    - booktabs        : Enhances tables with better-quality rules.
%%    - multirow        : Enables multi-row cells in tables.
%%    - mathlines       : Allows line numbering in math environments.
%%    - microtype       : Subtly adjust text (horizontal) spacing to reduce overfull boxes.
%%    - setspace        : Allows adjusting line spacing of specific texts
%%
%% Document Options:
%% 
%% 1. **old**: 
%%    - Activates a style that resembles the previous (legacy) Geoacta formatting.
%%    - Sets the document font to 12pt and adjusts layout with traditional section
%%      formatting and wider margins.
%% 
%% 2. **manuscrito**: 
%%    - Prepares the document for manuscript submission, separating the title page 
%%      and removing author/affiliation information on subsequent pages.
%%    - Sets line spacing to double and enables line numbering throughout.
%%    - Omits the acknowledgments section by default.
%%
%% 3. **nobilin**:
%%    - Enables a single-language mode for title, abstract, and keywords, using only 
%%      Spanish or English.
%%    - This option is ideal for documents that do not require bilingual content.
%%
%% 4. **en**:
%%    - Uses English labels (e.g., "Table" instead of "Tabla").
%%    - Recommended for documents entirely in English.
%%    - With both **nobilin** and **en**, only \title, \abstract, and \keywords 
%%      are used, ignoring the Spanish versions.
%%
%% 5. **nohead**: 
%%    - Suppresses headers, a setting also active by default in **manuscrito** mode.
%%
%% 6. **lineno/nolineno**:
%%    - Enables or disables line numbering across the document.
%%    - Line numbering is enabled by default in **manuscrito** mode.
%%
%% 7. **draft**:
%%    - Enables draft mode, useful for quick reviewing purposes. In this mode, figures 
%%      are replaced by empty rectangles, hyperlinks are disabled, and the bibliography 
%%      is omitted to speed up compilation.
%%
%% Key Commands:
%%
%% - \titulo{} and \title{} : Set the document title in Spanish and English, respectively. 
%%   Avoid capital letters unless necessary (e.g., for proper nouns). Place \titulo{} first, 
%%   followed by \title{} right after \begin{document}
%%   Example usage:
%%     \titulo{Análisis de datos sísmicos en Marte}
%%     \title{Seismic data analysis on Mars}
%%
%% - \autor{} : Defines the author(s) of the document, using the format: first name(s) 
%%   followed by last name, with numerical affiliation indicators as superscripts 
%%   This command should follow \title{}.
%%   Example usage:
%%     \autor{Carlos Gardel\textsuperscript{1} \& Alfredo Le Pera\textsuperscript{1,2}}
%%
%% - \afiliacion{} : Adds affiliation information for each author. Begin with 
%%   a numerical superscript indicator to match the authors’ affiliations.
%%   Multiple \afiliacion{} commands can be included as needed.
%%   Example usage:
%%     \afiliacion{\textsuperscript{1}Café de los Angelitos, Buenos Aires, Argentina.}
%%     \afiliacion{\textsuperscript{2}Paramount Studios, Joinville-le-Pont, Francia.}
%%
%% - \email{} : Provides the email address of the corresponding author. Should be used
%%   within the corresponding author's \afiliacion{} command  when **old** style is not
%%   selected.
%%   Example usage:
%%     \afiliacion{Carlos Gardel\textsuperscript{1}. \email{mibuenosairesquerido@gmail.com}}.
%%
%% - \contacto{} : Provides contact information, typically the email address of the 
%%   corresponding author, and should be used only in **old** style documents.
%%   Example usage:
%%     \contacto{mibuenosairesquerido@gmail.com}}.
%%
%% - \resumen{} : Sets the abstract in Spanish. Place this after \afiliacion{} or \contacto.
%%
%% - \abstract{} : Sets the abstract in English. Place this after \resumen{}.
%%
%% - \claves{} and \keywords{} : Set keywords in Spanish and English, respectively.
%%   \claves should be placed after \resumen{}, and \keywords after \abstract{}.
%%   The lists should be separated by commas, and capital letters should be avoided unless
%%   necessary (e.g., for proper nouns).
%%   Example usage:
%%     \claves{Análisis sísmico, Marte, datos geofísicos.}
%%     \keywords{Seismic analysis, Mars, geophysical data.}
%%
%% - \runninghead{}{} : Sets a short title and author(s) for the running header.
%%   Use the language of the document.
%%   Example usage:
%%     \runninghead{Short title}{Gardel}             % For one author (English)
%%     \runninghead{Título corto}{Gardel \& Le Pera} % For two authors (Spanish)
%%     \runninghead{Short title}{Gardel et al.}      % For three or more authors (English)
%%
%% - \pageone{} : Initializes the page numbering from the specified number. For example, 
%%   \pageone{5} will start page numbering at 5 and update the page range automatically.
%%
%% - \volumen{}{} : Sets the document’s volume and year. This command is optional; if not used,
%%   default values defined in geoacta.cls will be applied.
%%   Example usage:
%%     \volumen{47(2)}{2025} % Sets volume to 47(2) and year to 2025
%%
%% Recommended command order:
%% \titulo, \title, \autor, \afiliacion (as many as needed), \contacto (for **old** style),
%% \resumen, \claves, \abstract, \keywords. 
%% \runninghead, \pageone, and \volumen may be placed anywhere in the document. 
%%
%% Note: Not all commands are required. Most can be omitted, but the final result may 
%% not meet GEOACTA formatting requirements if certain elements are missing.
%%
%% Bibliography Style:
%%    - This class uses geoacta.bst, based on natbib, for managing citations 
%%      and references, compatible with both author-year and numeric citation styles.
%%
%% Ensure all necessary packages are available in your LaTeX distribution, or download 
%% from CTAN (www.ctan.org) if needed.
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

\NeedsTeXFormat{LaTeX2e}
\ProvidesClass{geoacta}[2024/11/02 v1.0 Geoacta Class]

% Load the base class
\LoadClass[a4paper]{book}

% Load common packages
\RequirePackage[utf8]{inputenc} % UTF-8 encoding
\RequirePackage[a4paper, headheight=2.2cm, headsep=0cm, footskip=1.8cm]{geometry} % Page layout settings
\RequirePackage{graphicx} % For including images
\RequirePackage{amsmath, amsfonts, amssymb} % For advanced mathematical typesetting
\RequirePackage[hyphens]{url} % URL handling
\RequirePackage[dvipsnames]{xcolor} % For including more colornames
%\RequirePackage[colorlinks=true, linkcolor=RoyalBlue, citecolor=RoyalBlue, urlcolor=RoyalBlue, breaklinks, pdfusetitle]{hyperref} % Hyperlink support
\RequirePackage[colorlinks=true, allcolors=RoyalBlue, breaklinks, pdfusetitle]{hyperref} % Hyperlink support
\RequirePackage{enumerate} % Enhanced enumeration options
\RequirePackage{tgheros} % Use sans-serif font similar to Helvetica
\renewcommand{\familydefault}{\sfdefault} % Set default font family to sans-serif
\RequirePackage[eulergreek]{sansmath} % Use sans-serif font for math mode
\sansmath % Apply sans-serif to math mode
\RequirePackage{titlesec} % For customizing section titles

\RequirePackage[backend=biber, style=apa, hyperref=true, uniquename=false, uniquelist=false,
  sortcites=false, maxnames=2, minnames=1]{biblatex}
\RequirePackage{csquotes,xpatch} % recommended for using with biblatex (?)

\RequirePackage{fancyhdr} % For custom headers and footers
\RequirePackage{lastpage} % For referencing the last page
\RequirePackage[mathlines]{lineno} % For line numbering in the document
\RequirePackage{etoolbox} % For advanced command patching
\RequirePackage{indentfirst} % For ensuring paragraph indentation after section headings
\RequirePackage{microtype} % Subtly adjust text spacing to reduce overfull boxes
\RequirePackage{setspace} % For adjusting the line spacing
\RequirePackage[outercaption]{sidecap} % To position captions on the side
\RequirePackage[labelfont={bf,it}, textfont=it, labelsep=period]{caption} % Caption formatting
\RequirePackage{booktabs} % For better table lines
\setlength\abovetopsep{-0.75em} % Reduce space between caption and table/figure (when caption is on top)
\RequirePackage{multirow} % For multi-row cells
\RequirePackage[detect-all,mode=text,table-number-alignment=right,
  table-text-alignment=center,separate-uncertainty=true,
  multi-part-units=single,table-align-text-pre=false]{siunitx} % For decimals alignment in tables (requires texlive-science)


% Enable line numbers for captions
\DeclareCaptionFont{linenumbers}{\internallinenumbers} % Declare a caption font with line numbers
\DeclareCaptionFont{nolinenumbers}{\nolinenumbers} % Declare a caption font with no line numbers

% Set line numbering font size
\renewcommand\linenumberfont{\normalfont\footnotesize\color{RoyalBlue}}  % Change 'tiny' to 'small', 'footnotesize', etc. for larger size

% Define a command to set the custom page number and page number range
\def\lastpage{\pageref*{LastPage}}  % \ref* instead of \pageref to avoid hyperlink for \lastpage
\def\pagerange{1--\lastpage} % default page range (in case \pageone is
% not set)
\newcommand{\pageone}[1]{\setcounter{page}{#1}\def\pagerange{#1--\lastpage}}

% Pre-title for Short note
\def\breve{{\Large Artículo breve / \textit{short note}} \\}

% Pre-title for Reporte
\def\reporte{{\Large Reporte / \textit{report}} \\}

% Set default values for \vol and \year in case \volumen is not used
\def\vol{46(1)}   % Default volume
\def\year{2024}   % Default year

% Define \volumen to set \vol and \year
\newcommand{\volumen}[2]{%
  \def\vol{#1}%
  \def\year{#2}%
}

% 
\newif\iffirstpage % Define a new boolean flag
\firstpagetrue     % Set it to true initially


% Patch 'normal' math environments for line numbering
\newcommand*\linenomathpatch[1]{%
    \cspreto{#1}{\linenomath}%
    \cspreto{#1*}{\linenomath}%
    \csappto{end#1}{\endlinenomath}%
    \csappto{end#1*}{\endlinenomath}%
}

% Apply the patch to various environments
\linenomathpatch{equation}
\linenomathpatch{gather}
\linenomathpatch{multline}
\linenomathpatch{align}
\linenomathpatch{alignat}
\linenomathpatch{flalign}

% Set the space between the figure and its caption
\captionsetup[SCfigure]{skip=0em} % Change 1em to your desired space

% Define boolean options for style selection
\newif\ifoldstyle % Option for old style
\newif\ifmanuscrito % Option for manuscript style
\newif\ifnobilin % Option for manuscript style
\newif\ifenglish % Option for articles in English language
\newif\ifnohead % Option for no header style
\newif\ifnolineno % Option for no linenumbers style
\newif\iflineno % Option for linenumbers style

% Custom command for choosing styles
\DeclareOption{old}{\oldstyletrue}
\DeclareOption{manuscrito}{\manuscritotrue}
\DeclareOption{nobilin}{\nobilintrue}
\DeclareOption{en}{\englishtrue}
\DeclareOption{nohead}{\noheadtrue}
\DeclareOption{nolineno}{\nolinenotrue}
\DeclareOption{lineno}{\linenotrue}
\DeclareOption*{\PassOptionsToClass{\CurrentOption}{book}} % Handle unknown options

% Process the options
\ProcessOptions\relax

\setlength{\parindent}{0em} % no indent for new paragraphs
\setlength{\parskip}{0.75em} % skip (almost) one line between paragraphs
\def\espaciado{1.05}

% Set document language
\ifenglish
  %\renewcommand{\bibsection}{\section{References}}
  \RequirePackage[english]{babel} % Language support
\else
  %\renewcommand{\bibsection}{\section{Referencias}}
  \RequirePackage[spanish,es-tabla]{babel} % Language support
  \selectlanguage{spanish}
  \decimalpoint
\fi


% Cleveref: references with links
\RequirePackage[nameinlink,noabbrev]{cleveref} % Global setting for cleveref

% Configuración automática de cleveref para español e inglés
\makeatletter
\@ifpackageloaded{babel}{
  \iflanguage{spanish}{
    \crefname{equation}{ecuación}{ecuaciones}
    \crefname{figure}{Figura}{Figuras}
    \crefname{table}{Tabla}{Tablas}
    \crefname{section}{Sección}{Secciones}
    \crefname{subsection}{Subsección}{Subsecciones}
    \crefname{chapter}{Capítulo}{Capítulos}
    \crefname{cb}{Recuadro}{Recuadros}

    \providecommand\crefpairconjunction{ y\nobreakspace}
    \providecommand\crefrangeconjunction{ a\nobreakspace}
    \providecommand{\crefmiddleconjunction}{,\nobreakspace}
    \providecommand{\creflastconjunction}{ y\nobreakspace}


    %\def\crefand{ y\nobreakspace}%
    %\def\crefmiddleconjunction{,\nobreakspace}%
    %\def\creflastconjunction{ y\nobreakspace}%
  }{
    \crefname{equation}{equation}{equations}
    \crefname{figure}{Figure}{Figures}
    \crefname{table}{Table}{Tables}
    \crefname{section}{Section}{Sections}
    \crefname{subsection}{Subsection}{Subsections}
    \crefname{chapter}{Chapter}{Chapters}
    \crefname{cb}{Box}{Boxes}

    %\def\crefand{ and\nobreakspace}%
    %\def\crefmiddleconjunction{,\nobreakspace}%
    %\def\creflastconjunction{ and\nobreakspace}%
  }
}{}
\makeatother

\creflabelformat{equation}{#1#2#3} % Mostrar ecuaciones sin paréntesis


% Section formatting based on selected style
\ifoldstyle
    \geometry{left=2.5cm, right=2.5cm, top=2.5cm, bottom=2.5cm} % Geometry settings for old style
    \renewcommand{\baselinestretch}{\espaciado} % Line spacing for text
    \renewcommand{\arraystretch}{\espaciado} % Line spacing for tables, etc.
    \renewcommand{\normalsize}{\fontsize{12}{14}\selectfont} % Font size for old style
    \def\sectionformat{\normalfont\bfseries\centering\MakeUppercase} % Section formatting for old style
\else
    \geometry{left=2.0cm, right=2.0cm, top=2.0cm, bottom=2.5cm} % Geometry settings for new style
    \renewcommand{\baselinestretch}{\espaciado} % Line spacing for text
    \renewcommand{\arraystretch}{\espaciado} % Line spacing for tables, etc.    
    \renewcommand{\normalsize}{\fontsize{11}{13}\selectfont} % Font size for new style
    %\renewcommand{\normalsize}{\fontsize{9}{10}\selectfont} % Font size for new style
    \def\sectionformat{\normalfont\bfseries\MakeUppercase} % Section formatting for new style
    %\def\sectionformat{\large\bfseries} % Section formatting for new style
\fi

% Section title formatting
\titleformat{\section}
  {\sectionformat} % Formatting for the section title
  {} % Section number
  {0em} % Space between the number and title
  {} % Code to execute after the section number (none in this case)

\titleformat{\subsection}
  {\normalfont\bfseries} % Formatting for the subsection title
  {} % Subsection number
  {0em} % Space between the number and title
  {} % Code to execute after the subsection number (none in this case)

\titleformat{\subsubsection}
  {\normalfont\it} % Formatting for the sub-subsection title
  {} % Sub-subsection number
  {0em} % Space between the number and title
  {} % Code to execute after the subsection number (none in this case)  

\titlespacing*{\section}
  {0pt} % No indentation from the left
  {1\baselineskip} % Space before the section title
  {0.5\baselineskip} % Space after the section title
  
\titlespacing*{\subsection}
  {0pt} % No indentation from the left
  {0.5\baselineskip} % Space before the subsection title
  {0.3\baselineskip} % Space after the subsection title

  
% Define commands for Title, Author, and Affiliation

\ifoldstyle
   \ifenglish
      \renewcommand{\title}[1]{\vspace*{-2em} 
      {\centering\Large\textbf{#1}\par} % Title in bold
      \def\mytitle{\title{#1}}} % Store the title for later use
      \newcommand{\titulo}[1]{}
   \else
      \newcommand{\titulo}[1]{\vspace*{-2em}
      {\centering\Large\textbf{#1}\par} % Title in bold
      \def\mititulo{\titulo{#1}}} % Store the title for later use
      \renewcommand{\title}[1]{}
   \fi
\else
   \ifnobilin
      \ifenglish
         \newcommand{\titulo}[1]{}
         \renewcommand{\title}[1]{
         {\setstretch{0.9}\parbox{\textwidth}{\raggedright\LARGE\textbf{#1}}} % Title in bold and allows wrapping
         \def\mytitle{\title{#1}}} % Store the title for later use     
      \else
         \newcommand{\titulo}[1]{
         {\setstretch{0.9}\parbox{\textwidth}{\raggedright\LARGE\textbf{#1}}} % Title in bold and allows wrapping
         \def\mititulo{\titulo{#1}}} % Store the title for later use
         \renewcommand{\title}[1]{}
      \fi
   \else    
      \newcommand{\titulo}[1]{
      {\setstretch{0.9}\parbox{\textwidth}{\raggedright\LARGE\textbf{#1}}} % Title in bold and allows wrapping
      \def\mititulo{\titulo{#1}}} % Store the title for later use
      \renewcommand{\title}[1]{
      {\setstretch{0.9}\parbox{\textwidth}{\vspace{1em}\raggedright\LARGE\textit{#1}}} % Title slanted and allows wrapping
      \def\mytitle{\title{#1}}} % Store the title for later use      
   \fi
\fi

% Define commands for Author and Affiliation

\ifoldstyle
   \newcommand{\autor}[1]{{\centering\textbf{#1}\par}} 
   \newcommand{\afiliacion}[1]{{\centering #1 \par}}  
\else
   \newcommand{\autor}[1]{
     {\parbox{\textwidth}{\vspace{1em}\raggedright\large #1}}} % Author formatting for new style                                                                                    
   \newcommand{\afiliacion}[1]{
   \setlength{\footnotesep}{0.3\baselineskip} % Adjust this value to control the spacing
   \let\thefootnote\relax % Remove the footnote number
   \footnotetext{\footnotesize #1}} % Add affiliation as footnote
  
   \makeatletter
   \renewcommand\@makefntext[1]{%
   \noindent\@makefnmark#1} % Remove indentation in footnotes
   \makeatother
\fi

   

% Define commands for Email/Contact

\ifoldstyle
   \newcommand{\contacto}[1]{{\centering Contacto: \href{mailto:#1}{#1} \par}}
   \newcommand{\email}[1]{}
\else
  \newcommand{\email}[1]{Email: \href{mailto:#1}{#1}}
  \newcommand{\contacto}[1]{}
\fi


% Define commands for Palabras clave/Keywords

\ifoldstyle
   \ifenglish
      \newcommand{\claves}[1]{}
      \newcommand{\keywords}[1]{\section{Keywords}#1}
   \else
      \newcommand{\claves}[1]{\section{Palabras clave}#1}
      \newcommand{\keywords}[1]{}
   \fi
\else
   \ifnobilin
      \ifenglish
         \newcommand{\claves}[1]{}
         \newcommand{\keywords}[1]{\vspace{-1em}\paragraph{Keywords}#1}
      \else
         \newcommand{\claves}[1]{\vspace{-1em}\paragraph{Palabras clave}#1}
         \newcommand{\keywords}[1]{}
      \fi
   \else
      \newcommand{\claves}[1]{\vspace{-1em}\paragraph{Palabras clave}#1}
      \newcommand{\keywords}[1]{\vspace{-1em}\paragraph{\textit{Keywords}}{\it #1}}
   \fi
\fi


% Define commands for Resumen/Abstract

\ifoldstyle
   \ifenglish
      \newcommand{\resumen}[1]{}
      \ifmanuscrito
         \newcommand{\abstract}[1]{\newpage\mytitle\section{Abstract}#1}
      \else
         \newcommand{\abstract}[1]{\section{Abstract}#1}
      \fi
   \else
      \newcommand{\abstract}[1]{}
      \ifmanuscrito
         \newcommand{\resumen}[1]{\newpage\mititulo\section{Resumen}#1}
      \else
         \newcommand{\resumen}[1]{\section{Resumen}#1}
      \fi
   \fi
\else
   \ifnobilin
      \ifenglish
         \newcommand{\resumen}[1]{}
         \ifmanuscrito
            \newcommand{\abstract}[1]{\newpage\mytitle\paragraph{Abstract}#1}
         \else
            \newcommand{\abstract}[1]{\paragraph{Abstract}#1}
         \fi
      \else
         \ifmanuscrito
            \newcommand{\resumen}[1]{\newpage\mititulo\paragraph{Resumen}#1}
         \else
            \newcommand{\resumen}[1]{\paragraph{Resumen}#1}
         \fi
         \newcommand{\abstract}[1]{}
      \fi
   \else
      \ifmanuscrito
         \newcommand{\resumen}[1]{\newpage\mititulo\mytitle\paragraph{Resumen}#1}
         \newcommand{\abstract}[1]{\paragraph{\textit{Abstract}}{\it #1}}
      \else
         \newcommand{\resumen}[1]{\paragraph{Resumen}#1}
         \newcommand{\abstract}[1]{\paragraph{\textit{Abstract}}{\it #1}}
      \fi
   \fi
\fi


% Agradecimientos
\ifoldstyle
  \ifenglish
    \newcommand{\gracias}[1]{\section{Acknowledgments}#1}
  \else
    \newcommand{\gracias}[1]{\section{Agradecimientos}#1}
  \fi
\else
  \ifenglish
    \newcommand{\gracias}[1]{\paragraph{Acknowledgments}#1}
  \else
    \newcommand{\gracias}[1]{\paragraph{Agradecimientos}#1}
  \fi
\fi


%
% Ajuste del string `version` para el idioma español:
%
% - Por defecto, `biblatex` traduce `version = {...}` como "Ver. ..."
%   lo cual no es compatible con el estilo APA 7 en español.
%
% - APA recomienda escribir "versión" completo, con minúscula, 
%   por ejemplo: "(versión 3.x)"
%
% - Este ajuste redefine el string `version` para que aparezca como "versión",
%   y así se respete la forma recomendada por APA 7 en español.
%
% DV, junio de 2025
\DefineBibliographyStrings{spanish}{
  version = {versión}
}



% Bibliografía (Danilo Velis, Junio 2025)

\ifenglish
  % Título en inglés
  \DefineBibliographyStrings{english}{
    bibliography = {References},
  }
\else
  % Título en español
  \DefineBibliographyStrings{spanish}{
  bibliography = {Referencias},
  andothers = {y otros},
  }
  % Esto reemplaza "&" por "y" en la lista de referencias
  % (En lugar de "\bibstring{and}" se puede poner " y", mismo resultado)
  \DeclareDelimFormat[bib]{finalnamedelim}{%
  \ifthenelse{\value{listcount}>\maxprtauth}
    {}
    {\ifthenelse{\value{liststop}>2}
       {\finalandcomma\addspace\bibstring{and}\space}
       {\addspace\bibstring{and}\space}}}
  %
  \DeclareDelimFormat[bib]{finalnamedelim:apa:family-given}{%
  \ifthenelse{\value{listcount}>\maxprtauth}
    {}
    {\ifthenelse{\ifcurrentname{groupauthor}\AND%
                 \value{liststop}=2}
     {\addspace\bibstring{and}\space}
     {\finalandcomma\addspace\bibstring{and}\space}}}
  % Esto reemplaza "&" por "y" en las cita en el texto
  % (En lugar de "\bibstring{and}" se puede poner " y", mismo resultado)
  \DeclareDelimFormat[parencite,nptextcite,fullcite,fullcitebib]{finalnamedelim}
  {\ifnum\value{liststop}>2 \finalandcomma\fi\addspace\bibstring{and}\space}
\fi

%
% Ajustes personalizados para cumplir con las normas del estilo APA 7:
%
% - En español, reemplaza "Consultado el..." por "Recuperado el..." al mostrar la
%   fecha de acceso (campo `urldate`).
% - Elimina la coma después de la fecha cuando el idioma activo es español
%   (según convenciones editoriales); en inglés, conserva la coma como lo requiere APA 7.
% - Reemplaza "desde" por "de" en la frase "Recuperado el... de [URL]".
% - Si no se especifica una fecha (`urldate`), se muestra únicamente el enlace (`url`),
%   sin texto adicional.
%
% Estos cambios se aplican tanto para entradas en español como en inglés, respetando
% las convenciones idiomáticas correspondientes.
%
% DV, junio de 2025


\DefineBibliographyStrings{spanish}{
  retrieved = {Recuperado},
  from      = {de},
}

\renewbibmacro*{url+urldate}{%
  \ifthenelse{\iffieldundef{url}\OR\NOT\iffieldundef{doi}}
    {}
    {\iffieldundef{urlyear}
      {}
      {%
        \iflanguage{english}
          {%
            \bibstring{retrieved}~\printurldate,~\bibstring{from}~%
          }
          {%
            \bibstring{retrieved}~\printurldate~\bibstring{from}~%
          }%
      }%
     \iffieldundef{url}{}{%
       \printfield{url}%
       \renewcommand*{\finentrypunct}{\relax}%
     }}}


%
% Ajuste para evitar corchetes automáticos alrededor del contenido del campo `note`.
%
% - Por defecto, el estilo APA envuelve `note={...}` entre corchetes.
% - Esta línea anula ese comportamiento y muestra el contenido tal como se ingresó:
%
% DV, junio de 2025

\DeclareFieldFormat{brackets}{#1}  % Anula los corchetes automáticos




% Ajuste del encabezado de la sección bibliográfica:
% - Elimina la numeración de la sección ("Referencias" o "Bibliografía")
% - Evita un salto de página automático
% - Muestra el título con el formato \section*{...}, sin incluirlo en el TOC
%
% DV, junio de 2025
\defbibheading{bibliography}[\bibname]{\section*{#1}}

% 
% Bibliography setup
%\setlength{\bibsep}{0.4em plus 0.3ex}
\setlength{\bibhang}{0em}
\setlength{\bibitemsep}{0.4em}


\ifoldstyle
   \renewcommand*{\bibfont}{\normalfont}
   \else
   \renewcommand*{\bibfont}{\small}
\fi   

% Header and footer setup
\pagestyle{fancy}
\renewcommand{\chaptermark}[1]{ \markboth{#1}{} }
\renewcommand{\sectionmark}[1]{ \markright{#1} }

\fancyhf{} % Clear existing header and footer styles

\ifoldstyle

  \fancyhead[LO,LE]{GEOACTA \vol: \pagerange\hspace{3pt}(\year)\\
    \textcopyright\hspace{3pt}\year{ A}sociación Argentina de Geofísicos y Geodestas}
  \newcommand{\runninghead}[2]{}

\else

   % Define a custom style for the first page
   \fancypagestyle{firstpagestyle}{%
   \fancyhf{} % Clear all headers and footers
   \fancyhead[L]{\it\small Geoacta \vol: \pagerange\hspace{3pt}(\year)}
   \fancyhead[R]{\it\small ISSN 1852-7744}
   \fancyfoot[L]{\small\hspace{5pt}\textcopyright\hspace{3pt}\sc
     \year{ A}sociación Argentina de Geofísicos y Geodestas}
   \fancyfoot[R]{\small\thepage}
   }
   % Set the first page style
   \AtBeginDocument{\thispagestyle{firstpagestyle}}

   % Set page style for subsequent pages
   
   % Command to set running heads for odd and even pages
   \newcommand{\runninghead}[2]{%
     \fancyhead[LE]{\small{\it #1}} % Left side for even pages
     \fancyhead[RO]{\small{\it #2}} % Right side for odd pages
   }

\fi
   
\fancyfoot[RO,LE]{\small\thepage} % Page number
\fancyfoot[RE,LO]{\small Geoacta \vol: \pagerange\hspace{3pt}(\year)} % Vol
%\fancyfoot[RE,LO]{\small\vol} % Volume

\setlength{\footskip}{1cm}  % Position page numbers 1cm from the bottom
\setlength{\headsep}{1em}  % Space between the rule and the text
\setlength{\headwidth}{\textwidth}  % Ensure header width matches text width

\renewcommand{\headrulewidth}{0.4pt}
\renewcommand{\footrulewidth}{0.0pt}


\ifmanuscrito
  % Suppress acknowledgments section in manuscript mode
  \renewcommand{\gracias}[1]{}
  
  % Set normal line spacing based on defined spacing variable
  \AtBeginDocument{\renewcommand{\baselinestretch}{\espaciado}
                   \renewcommand{\arraystretch}{\espaciado}
}
  
  % Set global double line spacing for manuscript
  \renewcommand{\baselinestretch}{2.0} % Line spacing

  % Ensure all tables have double spacing and preserve positioning options, including strict positioning
  \let\oldtable\table
  \let\endoldtable\endtable
  \renewenvironment{table}[1][htbp] % Default positioning to [htbp] if none specified
  {\begin{oldtable}[#1]\doublespacing} % Pass specified positioning options to original table environment
  {\end{oldtable}}
  
  % Apply double spacing to captions with a specific stretch factor
  \captionsetup{font={stretch=2.0}} 
  
  % Set double spacing and default font for bibliography
  \renewcommand*{\bibfont}{\normalfont\setstretch{2.0}}
  
  % Disable running head and header rule in manuscript mode
  \renewcommand{\runninghead}[2]{}
  \renewcommand{\headrulewidth}{0pt}
  
  % Enable line numbering throughout the document
  \linenumbers
  
  % Apply line numbering to captions in figures and tables
  \captionsetup[figure]{font+=linenumbers}
  \captionsetup[table]{font+=linenumbers}
\fi


\ifnohead
  \renewcommand{\runninghead}[2]{}{}
  \renewcommand{\headrulewidth}{0pt}
\fi

\iflineno
  \linenumbers
  \captionsetup[figure]{font+=linenumbers} % Apply line numbers to figure captions
  \captionsetup[table]{font+=linenumbers} % Apply line numbers to table captions
\fi
   
\ifnolineno
  \nolinenumbers
  \captionsetup[figure]{font+=nolinenumbers} % No line numbers to figure captions
  \captionsetup[table]{font+=nolinenumbers} % No line numbers to table captions
\fi


\endinput


